/* (C) 2024 YiRing, Inc. */
package com.yiring;

import cn.hutool.core.lang.Console;
import com.yiring.common.core.Retriever;
import org.junit.jupiter.api.Test;
import org.springframework.boot.SpringBootConfiguration;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.retry.annotation.EnableRetry;

@SpringBootConfiguration
@SpringBootTest
@EnableRetry
public class RetryTests {

    Retriever retriever = new Retriever();

    /**
     * 简易重试
     */
    @Test
    void def() {
        Integer value = retriever.execute(ctx -> {
            // eg: 可以在里面写需要重试的业务逻辑，例如：网络请求、数据库连接...

            Console.log("retry test, {}", ctx.getRetryCount());

            String text = "0.0";
            if (ctx.getRetryCount() > 1) {
                text = "1";
            }

            return Integer.parseInt(text);
        });
        Console.log("[default] retry test result: {}", value);
    }

    /**
     * 默认的退回策略，重试 3 次，每次间隔 1s
     */
    @Test
    void recover() {
        Integer value = retriever.execute(
            ctx -> {
                // eg: 可以在里面写需要重试的业务逻辑，例如：网络请求、数据库连接...

                Console.log("retry test, {}", ctx.getRetryCount());
                throw new RuntimeException("test");
            },
            ctx -> {
                // eg: 可以在重试超过指定次数和规则后，返回默认值，然后业务上可以根据返回值进行相应的处理，例如：返回 null，然后判断 null 时，发送告警消息

                Console.log("retry recover, {}", ctx.getRetryCount());
                return 666;
            },
            Retriever::defaultPolicy
        );
        Console.log("[recover] retry test result: {}", value);
    }

    /**
     * 重试 2 次，每次间隔 5s，失败默认退回 null
     */
    @Test
    void custom() {
        Integer value = retriever.execute(
            ctx -> {
                // eg: 可以在里面写需要重试的业务逻辑，例如：网络请求、数据库连接...

                Console.log("retry test, {}", ctx.getRetryCount());
                throw new RuntimeException("test");
            },
            template -> Retriever.setSimplePolicy(template, 2, 5000L)
        );
        Console.log("[custom] retry test result: {}", value);
    }
}
