/* (C) 2022 YiRing, Inc. */
package com.yiring.auth.web.user;

import com.github.xiaoymin.knife4j.annotations.ApiSupport;
import com.yiring.auth.domain.permission.Permission;
import com.yiring.auth.domain.permission.PermissionRepository;
import com.yiring.auth.domain.user.User;
import com.yiring.auth.util.Auths;
import com.yiring.auth.util.Permissions;
import com.yiring.auth.vo.permission.MenuVo;
import com.yiring.auth.vo.user.UserInfoVo;
import com.yiring.common.core.Result;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

/**
 * 系统用户控制器
 *
 * @author Jim
 * @version 0.1
 * 2022/1/24 14:13
 */

@Slf4j
@Validated
@ApiSupport(order = -9998)
@Tag(name = "用户信息", description = "UserView")
@RestController
@RequestMapping("/user/")
@RequiredArgsConstructor
public class UserViewController {

    final Auths auths;
    final PermissionRepository permissionRepository;

    @Operation(summary = "获取登录用户信息")
    @GetMapping("getUserInfo")
    public Result<UserInfoVo> getUserInfo() {
        User user = auths.getLoginUser();
        UserInfoVo userInfoVo = UserInfoVo
            .builder()
            .userId(user.getId())
            .mobile(user.getMobile())
            .username(user.getUsername())
            .realName(user.getRealName())
            .avatar(user.getAvatar())
            .roles(Permissions.toRoleVos(user.getRoles()))
            .build();
        return Result.ok(userInfoVo);
    }

    @Operation(summary = "获取用户菜单")
    @GetMapping("getMenuList")
    public Result<ArrayList<MenuVo>> getMenuList() {
        User user = auths.getLoginUser();

        // FIXED: admin 用户默认可以查询到所有菜单
        List<Permission> list;
        if (Auths.ADMIN_USER.equalsIgnoreCase(user.getUsername())) {
            list = permissionRepository.findAll();
        } else {
            list = Permissions.toPermissions(user.getRoles());
        }

        List<Permission> permissions = list
            .stream()
            .filter(permission -> !Permission.Type.BUTTON.equals(permission.getType()))
            .collect(Collectors.toList());
        List<MenuVo> vos = Permissions.toMenuTreeVo(permissions);
        return Result.ok((ArrayList<MenuVo>) vos);
    }

    @Operation(summary = "获取用户权限")
    @GetMapping("getPermCode")
    public Result<ArrayList<String>> getPermCode() {
        User user = auths.getLoginUser();
        List<Permission> permissions = Permissions.toPermissions(user.getRoles());
        List<String> codes = permissions.stream().map(Permission::getUid).collect(Collectors.toList());
        return Result.ok((ArrayList<String>) codes);
    }
}
